/*************************************************************************/
/*                                                                       */
/* Licensed Materials - Property of IBM                                  */
/*                                                                       */
/* (C) Copyright IBM Corp. 2010                                          */
/* All Rights Reserved                                                   */
/*                                                                       */
/* US Government Users Restricted Rights - Use, duplication or           */
/* disclosure restricted by GSA ADP Schedule Contract with IBM Corp.     */
/*                                                                       */
/*************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>
#include <string.h>
#include <libgen.h>
#include <unistd.h>
#include <sys/types.h>

#include <GL/glut.h>

#define REMOTE_VIEWER	1

#include "viewer.h"
#include "network_utils.h"


static int img_width, img_height, img_pixel_size, img_quality;
static GLenum img_format, img_type;

static size_t img_size;

static unsigned char *frame_buffer;

static int viewer_sock;
static int verbose = 0;


void * next_frame(int **size)
{
  int buffer_size;
  int ack_msg = htonl((uint32_t)VIEWER_MSG_FRAME_ACK);

  img_size = img_width * img_height * img_pixel_size;

  /* Receive the next buffer. They are always prefixed by the 
   * buffer size.
   */
  (void)block_recv(viewer_sock, &buffer_size, sizeof(buffer_size));
  buffer_size = (int)ntohl((uint32_t)buffer_size);

  (void)block_recv (viewer_sock, frame_buffer, img_size);
  
  /* Acknowledge the frame */
  (void)block_send (viewer_sock, &ack_msg, sizeof(ack_msg));

  if (size) *size = NULL;
  return (frame_buffer);
}


void mouse_action (int button, int state, int x, int y)
{
  struct _mouse_msg {
    uint32_t msg;
    viewer_mouse_action_t mouse;
  } mouse_msg;

  mouse_msg.msg          = htonl((uint32_t)VIEWER_MSG_MOUSE_ACTION);
  mouse_msg.mouse.button = htonl((uint32_t)button);
  mouse_msg.mouse.state  = htonl((uint32_t)state);
  mouse_msg.mouse.x      = htonl((uint32_t)x);
  mouse_msg.mouse.y      = htonl((uint32_t)y);

  (void)block_send (viewer_sock, &mouse_msg, sizeof(mouse_msg));
}

void mouse_motion (int x, int y)
{
  struct _mouse_msg {
    uint32_t msg;
    viewer_mouse_motion_t mouse;
  } mouse_msg;

  mouse_msg.msg          = htonl((uint32_t)VIEWER_MSG_MOUSE_MOTION);
  mouse_msg.mouse.x      = htonl((uint32_t)x);
  mouse_msg.mouse.y      = htonl((uint32_t)y);

  (void)block_send (viewer_sock, &mouse_msg, sizeof(mouse_msg));
}

void keyboard_action (unsigned char key, int x, int y)
{
  struct _key_msg {
    uint32_t msg;
    viewer_keyboard_action_t keyboard;
  } key_msg;

  key_msg.msg          = htonl((uint32_t)VIEWER_MSG_KEYBOARD_ACTION);
  key_msg.keyboard.key = htonl((uint32_t)key);
  key_msg.keyboard.x   = htonl((uint32_t)x);
  key_msg.keyboard.y   = htonl((uint32_t)y);

  (void)block_send (viewer_sock, &key_msg, sizeof(key_msg));
}

const char *usage =
  "Usage: viewer [-V] [-H] [-p <port_number>] [user@]<hostname> [<remote_application>]\n"
  "\n"
  "Launch the viewer for a remote application. The <hostname> specifies the host the\n"
  "remote application is running on, or to be run on. The viewer can also launch the\n"
  "remote OpenCL application by specifying the \"remote_application\", including path and\n"
  "parameters. The remote application is launched using ssh with the optionally specified\n"
  "\"user\".\n"
  "\n"
  "For example, the following invocation executes the viewer and remotely launches the\n"
  "julia sample on an accelerator device.\n"
  "\n"
  "  viewer host \"julia_remote -a\"\n"   
  "\n"
  "Options:\n"
  "  -H, --help               Display usage help.\n"
  "  -V, --verbose            Emit verbose informational output.\n"              
  "  -p, --port #             Connect to server via the specified port number. Default port\n"
  "                           if unspecified.\n"
  "\n"
  ;


int
main (int argc, char *argv[])
{
  int i;
  int port = 0;
  char* host;
  viewer_init_info_t info;
  int opt;
  char *optstring = "VHp:";
  static struct option long_options[] = {
    {"port", 1, NULL, 'p'},
    {"verbose", 0, NULL, 'V'},
    {"help", 0, NULL, 'H'},
    {NULL, 0, NULL, 0}
  };
  int remote_launch = 0;
  int remote_argc;
  char **remote_argv;
  pid_t remote_pid;

  /* Process input arguments 
   */
  while ((opt = getopt_long(argc, argv, optstring, long_options, NULL)) != -1) {
    switch (opt) {
    case 'p':
      port = atoi(optarg);
      break;
    case 'V':
      verbose = 1;
      break;
    case 'H':
    default:
      fprintf(stderr, usage);
      exit(EXIT_FAILURE);
      break;
    }
  }

  /* If user requested a remote program launch.
   */
  switch (argc - optind) {
  default:
    remote_launch = 1;
    /* Fall thru */
  case 1:
    if ((host = strchr(argv[optind], '@'))) {
      host++;
    } else {
      host = argv[optind];
    }
    break;
  case 0:
    fprintf(stderr, usage);
    exit(EXIT_FAILURE);
    break;
  }

  if (remote_launch) {
    /* Allocate memory for the ssh parameters */
    remote_argc = 3 + argc - optind;
    remote_argv = (char **)malloc(remote_argc*sizeof(char *));
    remote_argv[0] = "/usr/bin/ssh";
    remote_argv[1] = "-x";
    for (i=optind; i<argc; i++) {
      remote_argv[i+2-optind] = argv[i];
    }
    remote_argv[i+2-optind] = NULL;

    if (verbose) printf("Launching remote program %s on host %s\n", basename(remote_argv[2]), host);
    remote_pid = fork();
    if (remote_pid == 0) {
      if (execv(remote_argv[0], remote_argv) == -1) {
	perror("ERROR: Failed to remote exec program");
	exit(EXIT_FAILURE);
      }
    }
  }

  if (verbose) {
    if (port == 0) {
      printf ("Connecting to host, %s, over default port\n", host);
    } else {
      printf ("Connecting to host, %s, over port %d\n", host, port);
    }
  }

  viewer_sock = client_network_init((const char*)host, port);

  /* Fetch the image information
   */
  (void)block_recv(viewer_sock, &info, sizeof(info));

  img_width      = (int)ntohl(info.width);
  img_height     = (int)ntohl(info.height);
  img_pixel_size = (int)ntohl(info.pixel_size);
  img_format     = (GLenum)ntohl(info.format);
  img_type       = (GLenum)ntohl(info.type);
  img_quality    = (GLenum)ntohl(info.quality);

  img_size = img_width * img_height * img_pixel_size;

  if (verbose) {
    printf("Image Attributes: %d x %d x %d\n", img_width, img_height, img_pixel_size);
  }

  /* If the viewer has a diffent byte order than the client, then 
   * we can adjust the type / format to perform the byte swap within
   * OpenCL.
   */
  if (info.byte_order == VIEWER_IMAGE_BYTE_ORDER_REVERSED) {
    switch (img_type) {
    case GL_UNSIGNED_BYTE:
      /* No swap needed, this is a byte stream. */
      break;
    case GL_UNSIGNED_INT_8_8_8_8:
      img_type = GL_UNSIGNED_INT_8_8_8_8_REV;
      break;
    case GL_UNSIGNED_INT_8_8_8_8_REV:
      img_type = GL_UNSIGNED_INT_8_8_8_8;
      break;
    default:
      fprintf(stderr, "ERROR: Viewer does not currently support byte order reversing for the application's image type.\n");      
      break;
    }
  }

  /* Allocate a local frame buffer.
   */
  if ((frame_buffer = (unsigned char *)calloc (1, img_size)) == NULL) {
    fprintf(stderr, "INTERNAL ERROR: Failed to allocate framebuffer of %zd bytes\n", img_size);
    exit(EXIT_FAILURE);
  }

  // Init OpenGL viewer
  initViewer(info.name, img_width, img_height, img_pixel_size, img_format, img_type, 0);
  glutMainLoop();

  return 0;
}
