/*************************************************************************/
/*                                                                       */
/* Licensed Materials - Property of IBM                                  */
/*                                                                       */
/*                                                                       */
/*                                                                       */
/* (C) Copyright IBM Corp. 2010                                          */
/* All Rights Reserved                                                   */
/*                                                                       */
/* US Government Users Restricted Rights - Use, duplication or           */
/* disclosure restricted by GSA ADP Schedule Contract with IBM Corp.     */
/*                                                                       */
/*************************************************************************/

/*
 *====================================================================================
 * This file contains the server side source code for the remote viewer
 *====================================================================================
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "clock.h"
#include "viewer.h"
#include "network_utils.h"

/*-----------------------------------------------------------------------------
 *  GLOBAL VARIABLES and DEFINES
 *-----------------------------------------------------------------------------*/
static int img_size = 0;
static int img_width = 0;
static int img_height = 0;
static int viewer_sock;
static int frames_in_flight = 0;

/* This macro controls the maximum number of frames in flight if the 
 * server generates frames faster than the client can receive and display
 * them.
 */
#define MAX_FRAMES_IN_FLIGHT	5


/* 
 * ===  FUNCTION  ======================================================================
 *         Name:  initViewer
 *  Description:  initialize the GL viewer, which in this case it the glut environment 
 *                for displaying the resulting framebuffer
 *
 *  Parameters:
 *    name        String to the used to title the viewer window     
 *    width       The width, in pixels, of the image to be displayed
 *    height      The height, in pixels, of the image to be displayed
 *    pixel_size  The image's pixel size in bytes
 *    format      The GL pixel format of the image to be displayed.
 *    type        The GL pixel type of the image to be displayed.
 *    quality     The quality of the encoded images.
 * =====================================================================================
 */
void 
initViewer (const char *name, int width, int height, int pixel_size, GLenum format, GLenum type, int quality __attribute__ ((unused)))
{
  viewer_init_info_t info;

  img_width = width;
  img_height = height;
  img_size = width * height * pixel_size;

  /* Initialize the server network services, use the default port */
  viewer_sock = server_network_init(0);

  /* Construct the initial viewer information packet and send it. */
  (void)strncpy(info.name, name, VIEWER_MAX_NAME_LEN-1);
  info.name[VIEWER_MAX_NAME_LEN-1] = '\0';
  info.byte_order = VIEWER_IMAGE_BYTE_ORDER;
  info.width  = htonl((uint32_t)width);
  info.height = htonl((uint32_t)height);
  info.format = htonl((uint32_t)format);
  info.type   = htonl((uint32_t)type);
  info.pixel_size = htonl((uint32_t)pixel_size);
  info.quality = htonl((uint32_t)quality);

  (void)block_send(viewer_sock, &info, sizeof(info));
}

/* 
 * ===  FUNCTION  ======================================================================
 *         Name:  viewerMainLoop
 *  Description:  main loop for the remote viewer.
 * =====================================================================================
 */
void viewerMainLoop()
{
  int *size;
  uint32_t msg, frame_size;
  void *frame_buffer;

  while (1) {
    /* Process all of the available messages from the viewer 
     */
    while (block_recv_with_timeout(viewer_sock, &msg, sizeof(msg), 0.0)) {

      msg = ntohl(msg);

      switch (msg) {
      case VIEWER_MSG_MOUSE_ACTION:
	{
	  viewer_mouse_action_t mouse;
	  (void)block_recv(viewer_sock, &mouse, sizeof(mouse));
	  mouse_action((int)ntohl(mouse.button), (int)ntohl(mouse.state), (int)ntohl(mouse.x), (int)ntohl(mouse.y));
	}
	break;
      case VIEWER_MSG_KEYBOARD_ACTION:
	{
	  viewer_keyboard_action_t keyboard;
	  (void)block_recv(viewer_sock, &keyboard, sizeof(keyboard));
	  keyboard_action((unsigned char)ntohl(keyboard.key), (int)ntohl(keyboard.x), (int)ntohl(keyboard.y));
	}
	break;
      case VIEWER_MSG_MOUSE_MOTION:
	{
	  viewer_mouse_motion_t mouse;
	  (void)block_recv(viewer_sock, &mouse, sizeof(mouse));
	  mouse_motion((int)ntohl(mouse.x), (int)ntohl(mouse.y));
	}
	break;
      case VIEWER_MSG_FRAME_ACK:
	frames_in_flight--;
	break;
      default:
	fprintf(stderr, "INTERNAL ERROR: Unexpected msg of %d received\n", msg);
	exit(1);
	break;
      }
    }

    if (frames_in_flight < MAX_FRAMES_IN_FLIGHT) {
      /* Request the generation of the next frame, and send it to the viewer 
       */
      frame_buffer = next_frame(&size);
      frame_size = 0;
      (void)block_send(viewer_sock, &frame_size, sizeof(frame_size));
      (void)block_send(viewer_sock, frame_buffer, img_size);
      frames_in_flight++;
    }
  }
}
