/*************************************************************************/
/*                                                                       */
/* Licensed Materials - Property of IBM                                  */
/*                                                                       */
/*                                                                       */
/*                                                                       */
/* (C) Copyright IBM Corp. 2010                                          */
/* All Rights Reserved                                                   */
/*                                                                       */
/* US Government Users Restricted Rights - Use, duplication or           */
/* disclosure restricted by GSA ADP Schedule Contract with IBM Corp.     */
/*                                                                       */
/*************************************************************************/

#include <sys/types.h>
#include <sys/socket.h>
#include <stdio.h>
#include <stdlib.h>
#include <netinet/in.h>
#include <netdb.h>
#include <errno.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <sys/time.h>
#include <time.h>

#include "network_utils.h"

#define CLIENT_ID       	112358
#define SERVER_ID       	132639

#define NW_DEFAULT_PORT_NUMBER	60010

/* **************************************************************
 *                     INITIALIZATION ROUTINES                  *
 * **************************************************************
 */

/* 
 * ===  FUNCTION  ======================================================================
 *         Name:  client_network_init
 *  Description:  Initialize a network attached client. The client communicates with
 *                with the host specified by "host" over the port specified by "port". 
 *                The client must be initialized prior to the server.
 * =====================================================================================
 */
int client_network_init (const char* host, int port)
{
  int sock, client_id, ack;
  int sockval = 1;
  int retry_count = 0;
  struct sockaddr_in sin;
  struct hostent *hp;

  if (port == 0) port = NW_DEFAULT_PORT_NUMBER;

  /* Create the master socket.
   */
  if ((sock = socket (AF_INET, SOCK_STREAM, 0)) < 0) {
    perror("INTERNAL ERROR: Client could not open socket");
    exit(EXIT_FAILURE);
  }

  setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, (const void *) &sockval, (socklen_t)sizeof(sockval));

  /* Fill in the server's address and port into the sin structure.
   */
  memset(&sin, 0, sizeof(sin));
  sin.sin_family = AF_INET;
  hp = gethostbyname (host);
  memcpy(&sin.sin_addr, hp->h_addr, hp->h_length);
  sin.sin_port = htons (port);

  /* Connect to the server. No need to call bind. The OS will
   * assign a free port while performing the connect function.
   */
  while (connect (sock, (struct sockaddr *) &sin, sizeof (sin)) < 0) {
    if (retry_count++ == 1) {
      printf("Client is waiting to connect to server on port %d\n", port);
      fflush(stdout);
    }
    (void)sleep(1u);	/* Sleep for 1 second before trying again */
  }
  if (retry_count > 1) printf("Client successfully connected\n");

  /* Notify the server the client_id 
   */
  client_id = CLIENT_ID;
  client_id = htonl (client_id);
  if (send (sock, &client_id, sizeof(int), 0) < 0) {
    perror ("INTERNAL ERROR: Problem sending CLIENT_ID to server");
    close (sock);
    exit(EXIT_FAILURE);
  }
  
  /* Wait for acknowledgement that the server application program started.
   */
  if (recv (sock, &ack, sizeof(int), MSG_WAITALL) < 0) {
    perror ("INTERNAL ERROR: Problem with server application program acknowledgement");
    close (sock);
    exit(EXIT_FAILURE);
  }

  ack = ntohl (ack);
  if (ack != SERVER_ID) {
    fprintf (stderr, "INTERNAL ERROR: %s, receiving bad ack from server %d\n", __PRETTY_FUNCTION__, ack);
    close (sock);
    exit (EXIT_FAILURE);
  }
  return (sock);
}


/* 
 * ===  FUNCTION  ======================================================================
 *         Name:  server_network_init
 *  Description:  Initialize a network attached server. The server communicates with
 *                with the client over the port specified by "port". This routine must
 *                The serve must be initialized after the server.
 * =====================================================================================
 */
int server_network_init(int port)
{
  int sd, client_sock, cliLen;
  struct sockaddr_in cliAddr, servAddr;
  int id;
  int ack, rc;
      
  if (port == 0) port = NW_DEFAULT_PORT_NUMBER;

  /* create socket */
  sd = socket (AF_INET, SOCK_STREAM, 0);
  if (sd < 0) { 
    perror ("ERROR: Cannot open socket");
    exit(EXIT_FAILURE);
  } 
  int sockval = 1;
  setsockopt (sd, SOL_SOCKET, SO_REUSEADDR, (const void *) &sockval, (socklen_t)sizeof (sockval));
      
  /* bind server port */
  servAddr.sin_family = AF_INET;
  servAddr.sin_addr.s_addr = htonl (INADDR_ANY);
  servAddr.sin_port = htons (port);
  
  if (bind (sd, (struct sockaddr *) &servAddr, sizeof (servAddr)) < 0) {
    perror ("ERROR: Cannot bind port");
    exit(EXIT_FAILURE);
  }
  
  // Each server session can handle only one client  
  listen (sd, 1);

  printf ("Server waiting for client on port %d\n", port);

  cliLen = sizeof (cliAddr);
  client_sock = accept (sd, (struct sockaddr *) &cliAddr, (socklen_t*)&cliLen);
  if (client_sock < 0) {
    perror ("ERROR: Cannot accept connection");
    exit(EXIT_FAILURE);
  }

  if (fcntl (client_sock, F_SETFL, O_NONBLOCK) < 0) {
    perror ("ERROR: Failed to set socket to non_blocking mode");
    exit (EXIT_FAILURE);
  }
    
  // Receive an identification packet from client
  rc = block_recv (client_sock, &id, sizeof(int));

  if (rc != sizeof(int)) {
    fprintf (stderr, "INTERNAL ERROR: %s, error in receiving ID from client\n", __PRETTY_FUNCTION__);
    exit(EXIT_FAILURE);
  }
  id = ntohl (id);
  if (id != CLIENT_ID) {
    fprintf (stderr, "INTERNAL ERROR: %s, receiving bad ID from client %d\n", __PRETTY_FUNCTION__, id);
    exit (EXIT_FAILURE);
  }
    
  // Acknowledge that the identification packet

  ack = SERVER_ID;
  rc = send (client_sock, &ack, sizeof(int), 0);

  if (rc != sizeof(int)) {
    fprintf (stderr, "INTERNAL ERROR: %s, error in sending acknowlegement msg to client\n", __PRETTY_FUNCTION__);
    exit(EXIT_FAILURE);
  }

  return (client_sock);
}



/* ************************************************************* /
 *                     TIMING ROUTINES                           *
 * **************************************************************/
static struct timeval tp;
static long long num_secs;
static long long num_msecs;


static void start_timer (void)
{
  if (gettimeofday (&tp, NULL)) {
    fprintf (stderr, "INTERNAL ERROR in %s, call to gettimeoday failed\n", __PRETTY_FUNCTION__);
    exit (1);
  } 
  num_secs = tp.tv_sec;
  num_msecs = tp.tv_usec;
}

static double read_timer (void)
{
  double period;
 
  if (gettimeofday (&tp, NULL)) {
    fprintf (stderr, "INTERNAL ERROR in %s, call to gettimeoday failed\n", __PRETTY_FUNCTION__);
    exit (1);
  } 
  period = (double)((tp.tv_sec - num_secs)) * 1000000.0 + (double)((tp.tv_usec - num_msecs));
  return (period/1000000.0);
}



/* ************************************************************* /
 *                     SEND/RECEIVE ROUTINES                     *
 * **************************************************************/

/* 
 * ===  FUNCTION  ======================================================================
 *         Name:  block_recv_with_timeout
 *  Description:  Receive a block of data up to "n" bytes on the specified "stream". If the
 *                entire "n" bytes are not received with "timeout" seconds, the control
 *                is returned back to the caller. The number of bytes received is
 *                returned.
 * =====================================================================================
 */
size_t block_recv_with_timeout ( int stream, void *buffer, size_t n, double timeout )
{
  size_t  nleft = n;
  char *ptr = (char *) buffer;

  start_timer();
  while (nleft > 0) {
    ssize_t nreceived;

    if ((nreceived = recv(stream, ptr, nleft, 0)) < 0) {
      if ((errno == EINTR) || (errno == EWOULDBLOCK)) {
	nreceived = 0;
      } else {
	exit(0);
      }
    }
    nleft -= nreceived;
    ptr   += nreceived;
    
    if (read_timer() >= timeout) break;
  }
  return (n - nleft);
}

/* 
 * ===  FUNCTION  ======================================================================
 *         Name:  block_recv
 *  Description:  Receive a block of data up to "n" bytes on the specified "stream". 
 *                The number of bytes received is returned and is equal to "n" unless
 *                an unexpected error occurs.
 * =====================================================================================
 */
size_t block_recv ( int stream, void *buffer, size_t n)
{
  size_t nleft = n;
  char *ptr = (char *) buffer;

  while (nleft > 0) {
    ssize_t nreceived;

    if ((nreceived = recv(stream, ptr, nleft, 0)) < 0) {
      if ((errno == EINTR) || (errno == EWOULDBLOCK)) {
	nreceived = 0;
      } else {
	exit(0);
      }
    }
    nleft -= nreceived;
    ptr   += nreceived;
  }

  return (n - nleft);
}

/* 
 * ===  FUNCTION  ======================================================================
 *         Name:  block_send
 *  Description:  Send a block of data up to "n" bytes on the specified "stream". 
 *                The number of bytes sent is returned and is equal to "n" unless
 *                an unexpected error occurs.
 * =====================================================================================
 */

size_t block_send ( int stream, void *buffer, size_t n)
{
  size_t nleft = n;
  char *ptr = (char *)buffer;

  while (nleft > 0) {
    ssize_t nsent;

    if ((nsent = send(stream, ptr, nleft, 0)) < 0) {
      if ((errno == EINTR) || (errno == EWOULDBLOCK)) {
	nsent = 0;
      } else {
	exit(0);
      }
    }
    nleft -= nsent;
    ptr   += nsent;
  }
  return (n - nleft);
}
