/*************************************************************************/
/*                                                                       */
/* Licensed Materials - Property of IBM                                  */
/*                                                                       */
/*                                                                       */
/*                                                                       */
/* (C) Copyright IBM Corp. 2010                                          */
/* All Rights Reserved                                                   */
/*                                                                       */
/* US Government Users Restricted Rights - Use, duplication or           */
/* disclosure restricted by GSA ADP Schedule Contract with IBM Corp.     */
/*                                                                       */
/*************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "clock.h"
#include "viewer.h"

/*-----------------------------------------------------------------------------
 *  GLOBAL VARIABLES and DEFINES
 *-----------------------------------------------------------------------------*/

#define FRAME_RATE_UPDATE_PERIOD        1.0f    /* number of seconds between computing the frame rate */

static int gl_img_format, gl_img_type;          /* image format and type */
static int gl_img_width, gl_img_height;         /* image width and height */
static int vp_width, vp_height;                 /* viewport width and height */
static int frame_count = 0;                     /* current frame count */
static float frame_rate = 0.0f;
static char frame_rate_string[128] = { '\0' };
static void *frame_buffer = NULL;


#ifndef USE_DRAW_PIXELS
static GLuint texture;
#endif


/* 
 * ===  FUNCTION  ======================================================================
 *         Name:  display
 *  Description:  glutDisplayFunc - basically displays the frame buffer. Two display
 *                methods are included, either using DrawPixels or texture mapping. 
 *                Texture mapping has the advantage that linear filtering is employed
 *                when the window is resized from its default. The method of display
 *                used is defined at compiled time. The default is texture mapping.
 * =====================================================================================
 */
static void display (void)
{
  int i;

  if (frame_buffer) {
    /* Display the frame buffer image. 
     */
#ifdef USE_DRAW_PIXELS
    /* Render the image using glDrawPixels */
    glRasterPos2f (0.0f, 0.0f);
    glDrawPixels (gl_img_width, gl_img_height, gl_img_format, gl_img_type, frame_buffer);
#else
    /* Render the image using the texture engine */
    glTexSubImage2D(GL_TEXTURE_2D, 0, 0, 0, gl_img_width, gl_img_height, gl_img_format, 
                    gl_img_type, frame_buffer);
    glEnable(GL_TEXTURE_2D);
    glBegin(GL_TRIANGLE_STRIP);
    glTexCoord2i(0, 0);
    glVertex2i(0, 0);
    glTexCoord2i(0, 1);
    glVertex2i(0, vp_height);
    glTexCoord2i(1, 0);
    glVertex2i(vp_width, 0);
    glTexCoord2i(1, 1);
    glVertex2i(vp_width, vp_height);
    glEnd();
    glDisable(GL_TEXTURE_2D);
#endif /*  USE_DRAW_PIXELS */

    if (frame_rate != 0.0f) {
      glRasterPos2f (6.f, 6.f);
      for (i=0; i<strlen(frame_rate_string); i++) {
        glutBitmapCharacter (GLUT_BITMAP_HELVETICA_12, frame_rate_string[i]);
      }
    }
  } else {
    /* No frame buffer image available to display. Clear the window. */
    glClear (GL_COLOR_BUFFER_BIT);
  }

  glutSwapBuffers();
  glFlush();
}


/* 
 * ===  FUNCTION  ======================================================================
 *         Name:  reshape
 *  Description:  glut reshape function handler. Computes a new viewing transforn and
 *                adjust the pixel zoom.
 * =====================================================================================
 */
static void reshape (int w, int h)
{
  vp_width = w;
  vp_height = h;

#ifdef USE_DRAW_PIXELS
  glPixelZoom ((GLfloat)(w) / (GLfloat)(gl_img_width), (GLfloat)(h) / (GLfloat)(gl_img_height));
#endif

  glViewport (0, 0, (GLsizei)w, (GLsizei)h);
  glMatrixMode (GL_PROJECTION);
  glLoadIdentity ();
  glOrtho(0.0, (GLdouble)w, 0.0, (GLdouble)h, -1, 1);
  glMatrixMode(GL_MODELVIEW);

  glutPostRedisplay();

}

/* 
 * ===  FUNCTION  ======================================================================
 *         Name:  keyboard
 *  Description:  glut keyboard function handler. Processes keyboard actions to be 
 *                handled by the viewer. All other actions are passed to the 
 *                application's keyboard action function.
 *                The viewer handles the following key presses:
 *                1) escape - exit application
 * =====================================================================================
 */
static void keyboard(unsigned char key, int x, int y)
{
  switch (key) {
  case 27:              /* ESC */
    exit(EXIT_SUCCESS);
    break;
  default:
    /* Pass all other key actions to the application */
    keyboard_action(key, 
		    (int)((float)(x) * (float)(gl_img_width)  / (float)(vp_width)),
		    (int)((float)(y) * (float)(gl_img_height) / (float)(vp_height)));
    break;
  }
}

/* 
 * ===  FUNCTION  ======================================================================
 *         Name:  mouse
 *  Description:  glut mouse function handler. Processes mouse actions to be 
 *                handled by the viewer. The function transforms mouse positions
 *                as a result of window resizes so that applications need not be 
 *                aware of changes to the display window.
 * =====================================================================================
 */
void mouse(int button, int state, int x, int y)
{
  mouse_action(button, state, 
	       (int)((float)(x) * (float)(gl_img_width)  / (float)(vp_width)),
	       (int)((float)(y) * (float)(gl_img_height) / (float)(vp_height)));
}


/* 
 * ===  FUNCTION  ======================================================================
 *         Name:  motion
 *  Description:  glut mouse motion function handler. Processes mouse motion to be 
 *                handled by the viewer. The function transforms mouse positions
 *                as a result of window resizes so that applications need not be 
 *                aware of changes to the display window.
 * =====================================================================================
 */
void motion(int x, int y)
{
  mouse_motion((int)((float)(x) * (float)(gl_img_width)  / (float)(vp_width)),
	       (int)((float)(y) * (float)(gl_img_height) / (float)(vp_height)));
}



/* 
 * ===  FUNCTION  ======================================================================
 *         Name:  idle
 *  Description:  glut idle function handler. 
 * =====================================================================================
 */
static void idle()
{
  float interval;

  /* Request the next frame */
  frame_buffer = next_frame(NULL);
  frame_count++;

  /* Determine the current frame rate if it hasn't been determined or the 
   * update period has expired.
   */
  interval = stopclock();
  if ((interval > FRAME_RATE_UPDATE_PERIOD) || (frame_rate == 0.0f)) {
    frame_rate = (float)frame_count / interval;
    sprintf (frame_rate_string, "%.1f frames/sec", frame_rate);
    if (interval > FRAME_RATE_UPDATE_PERIOD) {
      frame_count = 0;
      startclock();
    }
  }
  glutPostRedisplay();
}

/* 
 * ===  FUNCTION  ======================================================================
 *         Name:  init_graphics
 *  Description:  initialize the OpenCL graphics rendering context to support the 
 *                display of the computed images.
 * =====================================================================================
 */
static void init_graphics()
{
  /* Initialize the graphics rendering context */

#ifndef USE_DRAW_PIXELS
  int w, h;

  glGenTextures(1, &texture);
  glBindTexture(GL_TEXTURE_2D, texture);
  glTexParameterf( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR );
  glTexParameterf( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR );
  glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);

  /* Round the texture map size to next power of 2 */
  for (w=1; w<gl_img_width; w<<=1);
  for (h=1; h<gl_img_height; h<<=1);

  glTexImage2D(GL_TEXTURE_2D, 0, 4, w, h, 0, gl_img_format, gl_img_type, NULL);

  glMatrixMode(GL_TEXTURE);
  glLoadIdentity();
  glScalef((GLfloat)(gl_img_width)/(GLfloat)(w), (GLfloat)(gl_img_height)/(GLfloat)(h), 1.0f);
  glMatrixMode(GL_MODELVIEW);
#endif /* USE_DRAW_PIXELS */

  glClearColor (0.0f, 0.0f, 0.0f, 1.0f);
  glDrawBuffer (GL_BACK);
  glDisable(GL_DEPTH_TEST);
}



/* 
 * ===  FUNCTION  ======================================================================
 *         Name:  initViewer
 *  Description:  initialize the GL viewer, which in this case it the glut environment 
 *                for displaying the resulting framebuffer
 *
 *  Parameters:
 *    name        String to the used to title the viewer window     
 *    width       The width, in pixels, of the image to be displayed
 *    height      The height, in pixels, of the image to be displayed
 *    pixel_size  The image's pixel size in bytes
 *    format      The GL pixel format of the image to be displayed.
 *    type        The GL pixel type of the image to be displayed.
 *    quality     The encoded image quality. Unused by the GL viewer.
 * =====================================================================================
 */
void 
initViewer (const char *name, int width, int height, int pixel_size, GLenum format, GLenum type, int quality __attribute__ ((unused)))
{
  int argc = 1;

  gl_img_width = width;
  gl_img_height = height;

  if (getenv("GL_VIEWER_BYTE_SWAP")) {
    switch (format) {
    case GL_RGBA:
      format = GL_ABGR_EXT;
      break;
    case GL_ABGR_EXT:
      format = GL_RGBA;
      break;
    case GL_RGB:
      format = GL_BGR;
      break;
    case GL_BGR:
      format = GL_RGB;
      break;
    }
  }
  gl_img_format = format;
  gl_img_type = type;
  frame_count = 0;

  glutInit (&argc, (char **)&name);
  glutInitDisplayMode (GLUT_DOUBLE | GLUT_RGBA);
  glutInitWindowSize (gl_img_width, gl_img_height);
  glutInitWindowPosition (32, 32 + (1024 - gl_img_width));
  glutCreateWindow (name);
  glutDisplayFunc (display);
  glutReshapeFunc (reshape);
  glutKeyboardFunc (keyboard);
  glutMouseFunc (mouse);
  glutMotionFunc (motion);
  glutIdleFunc (idle);

  init_graphics();
  startclock();
}

/* 
 * ===  FUNCTION  ======================================================================
 *         Name:  viewerMainLoop
 *  Description:  main loop for the GL viewer. Since this viewer uses GLUT, it calls 
 *                glutMainLoop.
 * =====================================================================================
 */
void viewerMainLoop()
{
  glutMainLoop();
}
