#!/usr/bin/perl

#############################################################################
#
# Licensed Materials - Property of IBM
#
#
# (C) Copyright IBM Corp. 2009
# All Rights Reserved
#
# US Government Users Restricted Rights - Use, duplication or
# disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
#
#############################################################################


#############################################################################
# Overview:
#   This script runs through all the different command line parameter options
#   that the Black-Scholes sample offers and determines which of these option
#   combinations would produce the best performance run on a particular 
#   OpenCL platform. 
#
#   The output of this script provides performance information on the following:
#   	_ The best and worst command and the associated measured time for 
#   	  setting up the sample. This includes parsing the command line 
#   	  arguments and setting up the OpenCL platform, context, command queue(s),
#   	  and device(s). 
#
#   	_ The best and worst command and the associated measured time for
#   	  initializing data for the sample. This includes allocating the 
#   	  necessary data buffers on the host (if any), creating the OpenCL 
#   	  memory buffer objects, and intialize the buffers with a set of 
#   	  reasonable starting values
#
#       _ The best and worst command and the associated measured time for
#         executing the kernel. This includes enqueueing the kernel on the 
#         appropriate OpenCL Command queue(s) and wait for the execution to 
#         finish
#
#       _ The best and worst command and the associated measured time for
#         shuting down the sample. This includes freeing up resources and 
#         releasing OpenCL resources.
#
#       _ The best and worst overall command and the associated accumulated
#         time for running the entire sample. 
#
#   For a detailed description of all the possible input parameters to the 
#   sample Black-Scholes, please look at the readme.blackscholes.txt file
#
# Parameters:
#   The script accepts two parameters, the first one is mandatory and the 
#   second one is optional
#
#   first parameter - device_type:  the device type input parameter can be one
#   of the followings:
#   	cpu:
#   	gpu:
#   	accel:
#
#   second parameter - output_filename: All output will be logged in this file. 
#   If the user does not provide a filename, outputs will be displayed on 
#   stdout and stderr and will not be logged. 
#############################################################################


#We are using strict and warnings to make sure we are not making hard-to-track
#down bugs
use strict;
use warnings;

#Variable declarations

#These are the different input parameters
my @kernel_options = ("taskDB", "taskSB", "taskLS", "rangeLS", "rangeAWGC");
my @vector_width_options = ("1", "2", "4", "8", "16");
my @lwgsize_options = ("1", "2", "4", "8", "16", "32", "64", "128");
my @buffer_options = ("none", "use", "alloc", "alloc_copy", "copy");

#Variables for setting up the different command runs
my @command_output = ("");
my $command_result = 0;
my $line = "";
my $run_flag = 1;
my $command = "";
my $num_runs = 0;

#Variables to store different command line options
my $kernel_option = "";
my $buffer_option = "";
my $vector_width_option = "";
my $lwgsize_option = "";
my $arraysize_option = 4194304;

#Variables for per command run timing
my $data_init_time = 0;
my $kernel_exec_time = 0;
my $shutdown_time = 0;
my $kernel_build_time = 0;
my $setup_time = 0;

#Variables for kernel execution statistics
my $best_kernel_exec_time = 32000.0;
my $worst_kernel_exec_time = 0.0;
my $best_kernel_exec_command = "";
my $worst_kernel_exec_command = "";
my $avg_kernel_exec_time = 0.0;

#Variables for data initialization statistics
my $best_data_init_time = 32000.0;
my $worst_data_init_time =  0.0;
my $avg_data_init_time = 0.0;
my $best_data_init_command = "";
my $worst_data_init_command = "";

#Variables for shutdown statistics
my $best_shutdown_time = 32000.0;
my $worst_shutdown_time = 0.0;
my $avg_shutdown_time = 0.0;
my $best_shutdown_command = "";
my $worst_shutdown_command = "";

#Variables for setup statistics
my $best_setup_time = 32000.0;
my $worst_setup_time = 0.0;
my $avg_setup_time = 0.0;
my $best_setup_command = "";
my $worst_setup_command = "";

#Variables for overall sample statistics
my $total_time = 0;
my $best_total_time = 320000.0;
my $worst_total_time = 0.0;
my $avg_total_time = 0.0;
my $best_total_command = "";
my $worst_total_command = "";

#input parameters
my $device = $ARGV[0];
my $output_file = "NO_OUTPUT_FILE";
my $use_outfile = 0;

#Function to print out the usage
sub help {
print STDOUT <<EOF;
  $0 <device_type> <output_file>
  	<device_type> - type of the device you want to run blackscholes on
	                can be one of the following 3 options: cpu, gpu, or accel
	<output_file> - all outputs are going to be logged in this file. If this
	                file is not specified, output will not be logged  
EOF
exit 1;
}

#Ensuring that we have at least one input parameter
if ($#ARGV < 0 || $ARGV[0] eq "--help" || $ARGV[0] eq "-h" || $ARGV[0] eq "-?") {
  &help;
}


if ($#ARGV >= 1)
{
  $output_file = $ARGV[1];
  open (OUT, "> $output_file") || print "Could not open file $output_file for writing: $!. Output will not be logged\n";
  $use_outfile = 1;
}

print "Running blackscholes on $device device with output_file = $output_file\n";
if ($use_outfile == 1)
{
  print OUT "Running blackscholes on $device device with output_file = $output_file\n";
}

if (($device ne "cpu") && ($device ne "gpu") && ($device ne "accel"))
{
  &help;
}

#Looping over all the different command line options
foreach $kernel_option (@kernel_options)
{
  foreach $vector_width_option (@vector_width_options)
  {
    foreach $buffer_option (@buffer_options)
    {
      foreach $lwgsize_option (@lwgsize_options)
      {
	#if the selected kernel is rangeLS or rangeAWGC, then we really loop over the $lwgsize, 
	#otherwise, we can safely ignore the $lwgsize option
	if (($kernel_option eq "rangeLS") || ($kernel_option eq "rangeAWGC"))
	{
	  $command = "./bsop --$device --arraysize $arraysize_option --buffer $buffer_option --$kernel_option --vectorwidth $vector_width_option --lwgsize $lwgsize_option";
	}
	else
	{
	  if ($lwgsize_option ne "1")
	  {
	    $run_flag = 0;
	  }
	  else
	  {
	    $command = "./bsop --$device --arraysize $arraysize_option --buffer $buffer_option --$kernel_option --vectorwidth $vector_width_option";
	  }
	}

	#if we're actually running the command
	if ($run_flag == 1)
	{
	  print "\nPERL Running Command = $command\n";
	  @command_output = `$command 2>&1`;
	  $command_result = `echo $?`;
	  $num_runs = $num_runs + 1;

	  #Check the return code
	  if ($command_result != 0)
	  {
	    print "\"$command\" FAILED\n";
	    if ($use_outfile == 1)
	    {
	      print OUT "\"$command\" FAILED\n";
	      print OUT "@command_output";
	    }
	  }
	  else
	  {
	    if ($use_outfile == 1)
	    {
	      print OUT "\"$command\" PASSED\n";
	      print OUT "@command_output";
              print OUT "\n";
	    }

	    foreach $line (@command_output)
	    {
	      if ($line =~ m/setup time/)
	      {
		$setup_time = $line;
		$setup_time =~ s/setup time: //;
		$setup_time =~ s/\n//;
	      }

	      if ($line =~ m/data initialization time/)
	      {
		$data_init_time = $line;
		$data_init_time =~ s/data initialization time: //;
		$data_init_time =~ s/\n//;
	      }

	      if ($line =~ m/kernel build time/)
	      {
		$kernel_build_time = $line;
		$kernel_build_time =~ s/kernel build time: //;
		$kernel_build_time =~ s/\n//;
	      }

	      if ($line =~ m/kernel execution time/)
	      {
		$kernel_exec_time = $line;
		$kernel_exec_time =~ s/kernel execution time: //;
		$kernel_exec_time =~ s/\n//;
	      }

	      if ($line =~ m/shutdown time/)
	      {
		$shutdown_time = $line;
		$shutdown_time =~ s/shutdown time: //;
		$shutdown_time =~ s/\n//;
	      }

	    } #end foreach $line in the STDOUT output

	    $total_time = $setup_time + $data_init_time + $kernel_exec_time + 
	                  + $shutdown_time;
			  
	    if ($total_time < $best_total_time)
	    {
	      $best_total_time = $total_time;
	      $best_total_command = $command;
	    }

	    if ($total_time > $worst_total_time)
	    {
	      $worst_total_time = $total_time;
	      $worst_total_command = $command;
	    }

	    if ($kernel_exec_time < $best_kernel_exec_time)
	    {
	      $best_kernel_exec_time = $kernel_exec_time;
	      $best_kernel_exec_command = $command;
	    }

	    if ($kernel_exec_time > $worst_kernel_exec_time)
	    {
	      $worst_kernel_exec_time = $kernel_exec_time;
	      $worst_kernel_exec_command = $command;
	    }

	    if ($shutdown_time < $best_shutdown_time)
	    {
	      $best_shutdown_time = $shutdown_time;
	      $best_shutdown_command = $command;
	    }

	    if ($shutdown_time > $worst_shutdown_time)
	    {
	      $worst_shutdown_time = $shutdown_time;
	      $worst_shutdown_command = $command;
	    }
	    if ($data_init_time < $best_data_init_time)
	    {
	      $best_data_init_time = $data_init_time;
	      $best_data_init_command = $command;
	    }

	    if ($data_init_time > $worst_data_init_time)
	    {
	      $worst_data_init_time = $data_init_time;
	      $worst_data_init_command = $command;
	    }
	    if ($setup_time < $best_setup_time)
	    {
	      $best_setup_time = $setup_time;
	      $best_setup_command = $command;
	    }

	    if ($setup_time > $worst_setup_time)
	    {
	      $worst_setup_time = $setup_time;
	      $worst_setup_command = $command;
	    }

	    $avg_kernel_exec_time = $avg_kernel_exec_time + $kernel_exec_time;
	    $avg_data_init_time = $avg_data_init_time + $data_init_time;
	    $avg_shutdown_time = $avg_shutdown_time + $shutdown_time;
	    $avg_setup_time = $avg_setup_time + $setup_time;
	    $avg_total_time = $avg_total_time + $total_time;

	  } #end else statement - if the program is PASSING
	} #end if we are runnning the command
	else
	{
	  $run_flag = 1;
	}
      } #end lwgsize
    } #end for loop buffer option

  } #end forloop vector_width option

} #end for loop kernel option

#Find the avg times
$avg_kernel_exec_time = $avg_kernel_exec_time/$num_runs;
$avg_data_init_time = $avg_data_init_time/$num_runs;
$avg_shutdown_time = $avg_shutdown_time/$num_runs;
$avg_setup_time = $avg_setup_time/$num_runs;
$avg_total_time = $avg_total_time/$num_runs;

#Print statistics
if ($use_outfile == 1)
{
  print OUT  "======================================================================================\n";
  print OUT  "\nOn this device ($device):\n";
  print OUT  "\tThe best command for setup is $best_setup_command with setup time = $best_setup_time\n";
  print OUT  "\tThe worst command for setup is $worst_setup_command with setup time = $worst_setup_time\n";
  print OUT  "\tThe average kernel for setup execution time is $avg_setup_time\n\n";

  print OUT  "\tThe best command for data initialization is $best_data_init_command, time is $best_data_init_time\n";
  print OUT  "\tThe worst command for data initialization is $worst_data_init_command, time is $worst_data_init_time\n";
  print OUT  "\tThe average data initialization time is $avg_data_init_time\n\n";

  print OUT  "\tThe best command for kernel execution is $best_kernel_exec_command with kernel exec time = $best_kernel_exec_time\n";
  print OUT  "\tThe worst command for kernel execution is $worst_kernel_exec_command with kernel exec time = $worst_kernel_exec_time\n";
  print OUT  "\tThe average kernel execution execution time is $avg_kernel_exec_time\n\n";

  print OUT  "\tThe best command for shutdown is $best_shutdown_command, time is $best_shutdown_time\n";
  print OUT  "\tThe worst command for shutdown is $worst_shutdown_command, time is $worst_shutdown_time\n";
  print OUT  "\tThe average shutdown time is $avg_shutdown_time\n\n";

  print OUT  "\tThe best command overall is $best_total_command with time measured = $best_total_time\n";
  print OUT  "\tThe worst command overall is $worst_total_command with time measured = $worst_total_time\n";
  print OUT  "\tThe average time for total execution time is $avg_total_time\n\n";
  print OUT  "======================================================================================\n";
}

print  "======================================================================================\n";
print  "\nOn this device ($device):\n";
print  "\tThe best command for setup is $best_setup_command with setup time = $best_setup_time\n";
print  "\tThe worst command for setup is $worst_setup_command with setup time = $worst_setup_time\n";
print  "\tThe average kernel for setup execution time is $avg_setup_time\n\n";

print  "\tThe best command for data initialization is $best_data_init_command, time is $best_data_init_time\n";
print  "\tThe worst command for data initialization is $worst_data_init_command, time is $worst_data_init_time\n";
print  "\tThe average data initialization time is $avg_data_init_time\n\n";

print  "\tThe best command for kernel execution is $best_kernel_exec_command with kernel exec time = $best_kernel_exec_time\n";
print  "\tThe worst command for kernel execution is $worst_kernel_exec_command with kernel exec time = $worst_kernel_exec_time\n";
print  "\tThe average kernel execution execution time is $avg_kernel_exec_time\n\n";

print  "\tThe best command for shutdown is $best_shutdown_command, time is $best_shutdown_time\n";
print  "\tThe worst command for shutdown is $worst_shutdown_command, time is $worst_shutdown_time\n";
print  "\tThe average shutdown time is $avg_shutdown_time\n\n";

print  "\tThe best command overall is $best_total_command with time measured = $best_total_time\n";
print  "\tThe worst command overall is $worst_total_command with time measured = $worst_total_time\n";
print  "\tThe average time for total execution time is $avg_total_time\n\n";
print  "======================================================================================\n";

if ($use_outfile == 1)
{
  close (OUT);
}

