/*************************************************************************/
/*                                                                       */
/* Licensed Materials - Property of IBM                                  */
/*                                                                       */
/* (C) Copyright IBM Corp. 2009                                          */
/* All Rights Reserved                                                   */
/*                                                                       */
/* US Government Users Restricted Rights - Use, duplication or           */
/* disclosure restricted by GSA ADP Schedule Contract with IBM Corp.     */
/*                                                                       */
/*************************************************************************/
/* PROLOG END TAG zYx                                                    */

/*************************************************************************/
/*                                                                       */
/* This utility will call the appropriate OpenCL API calls to build a    */
/* kernel binary for the devices specified.                              */
/*                                                                       */
/*************************************************************************/

#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <CL/cl.h>
#include <getopt.h>
#include <errno.h>

#include "check.h"

/**
 * process_command_line
 * 
 * Parse the command line options and return the option values.
 */
int process_command_line(int argc, char** argv, 
                         cl_device_type *dev_type, char **flags, char
                         **output, char **input, int *quiet) 
{

  char *usage =
    "Usage: %s [DEVICE] [OPTIONS...] [FILE]\n"
    "\n"
    "Build OpenCL kernel binary for specified device type from the specified FILE.\n"
    "With no FILE, or when FILE is -, source will be read from standard input.\n"
    "\n"
    " Device Types: (only specify one)\n"
    "\n"
    "  -a, --accel              build for CL_DEVICE_TYPE_ACCELERATOR\n"
    "  -c, --cpu                build for CL_DEVICE_TYPE_CPU\n"              
    "  -A, --all                build for CL_DEVICE_TYPE_ALL\n"              
    "  -d, --default            build for CL_DEVICE_TYPE_DEFAULT (default)\n"              
    "\n"
    " Options:\n"
    "\n"
    "  -f, --flags \"options\"    OpenCL Buld option flags (default: none)\n"
    "  -o, --output <filebase>  output binary filebase (default output <source_name>_<DEVICE NAME>.ocl\n"
    "  -q, --quiet              no output (default: not quiet)\n"
    "  -h, --help               display usage information and exit\n"
    "\n"
    ;

  while (1) {
    static struct option long_options[] = {
        {"accel", 0, NULL, 'a'},
        {"cpu", 0, NULL, 'c'},
        {"all", 0, NULL, 'A'},
        {"default", 0, NULL, 'd'},
        {"flags", 1, NULL, 'f'},
        {"output", 1, NULL, 'o'},
        {"quiet", 0, NULL, 'q'},
        {"help", 0, NULL, 'h'},
        {NULL, 0, NULL, 0}
    };

    int option_index = 0;
    int opt = getopt_long(argc, argv, "acAdf:o:qh", long_options, &option_index);
    
    if (opt == -1) {
      break;
    }
    
    switch (opt) {
    case 'a':
      *dev_type = CL_DEVICE_TYPE_ACCELERATOR;
      break;
    case 'c':
      *dev_type = CL_DEVICE_TYPE_CPU;
      break;
    case 'A':
      *dev_type = CL_DEVICE_TYPE_ALL;
      break;
    case 'd':
      *dev_type = CL_DEVICE_TYPE_DEFAULT;
      break;
    case 'f':
       *flags = optarg;
      break;
    case 'o':
      *output = optarg;
      break;
    case 'q':
      *quiet = 1;
      break;
    case 'h':
      fprintf(stdout, usage, argv[0]);
      exit(EXIT_SUCCESS);
      break;
    default:
      return 1;
    } // switch
  } // while

  if (argc > (optind + 1)) {
    fprintf(stderr, "ERROR: expecting only 1 FILE\n");
    return 1;
  }

  if (optind < argc) {
    // get the source name
    *input = argv[optind];
  }

  return 0;
}


/**
 * main
 * 
 * Find the source and build the binary
 */
int main(int argc, char** argv)
{
  int rc;
  size_t size_ret;
  int fin;
  char *source;
  size_t s_size;
  uint num_devices;
  int dev;

  // variables to track things that the user can change.
  cl_device_type dev_type = CL_DEVICE_TYPE_DEFAULT;
  char *source_filename = NULL;
  char *binary_filename = NULL;
  char *options = NULL;
  int quiet = 0;

  // OpenCL objects
  cl_context context;             // compute context
  cl_program program;             // compute program

  // Process command line arguments
  //
  if (process_command_line(argc, argv, &dev_type, &options, &binary_filename, &source_filename, &quiet)) {
    exit(EXIT_FAILURE);
  }
  
  if (source_filename && !strcmp(source_filename,"-"))  {
    source_filename = NULL;
  }

  // get the source code, either from stdin or from a file
  if (source_filename == NULL) {
    // standard input
    fin = STDIN_FILENO;

    int in_size = 0;
    int read_size;

    // start out with 1024
    source = malloc(1024 * sizeof(char));
    if (source == NULL) {
      PRINT(stderr, "ERROR: malloc for source buffer failed\n");
      exit(EXIT_FAILURE);
    }

    // read the data in
    read_size = read(fin, source, 1024);
    if (read_size == -1) {
      PRINT(stderr, "ERROR: read failed errno=%d\n", errno);
      exit(EXIT_FAILURE);
    }

    in_size += read_size;
    while (read_size > 0) {
      // get more room and read more data
      source = realloc(source, in_size + (1024 * sizeof(char)));
      if (source == NULL) {
        PRINT(stderr, "ERROR: realloc for source buffer failed\n");
        exit(EXIT_FAILURE);
      }

      read_size = read(fin, source + in_size, 1024);
      if (read_size == -1) {
        PRINT(stderr, "ERROR: read failed errno=%d\n", errno);
        exit(EXIT_FAILURE);
      }

      in_size += read_size;
    } // while

  } else {
    // not stdin; now try to open the source file
    fin = open(source_filename, O_RDONLY);
    if (fin == -1) {
      PRINT(stderr, "ERROR: open(\"%s\",O_RDONLY) failed errno=%d\n", source_filename, errno);
      exit(EXIT_FAILURE);
    }

    // and mmap it so that we can send the source to OpenCL
    struct stat statbuf;
    stat (source_filename, &statbuf);
    s_size = (size_t) statbuf.st_size;
    source = (char*)mmap(NULL, s_size, PROT_READ, MAP_PRIVATE, fin, 0);
    if (source == NULL) { 
      PRINT(stderr, "ERROR: mmap of source failed; errno=%d\n", errno);
      exit(EXIT_FAILURE);
    }
  }

  // start with OpenCL work

  // get the desired device(s) - how many are there for this type?
  CHECK_ERROR(clGetDeviceIDs(NULL, dev_type, 0, NULL, &num_devices))

  // get id's for those devices
  cl_device_id device_ids[num_devices];
  CHECK_ERROR(clGetDeviceIDs(NULL, dev_type, num_devices, device_ids, NULL))

  // grab the name of each device, for future use
#define MAX_FILENAME_LEN 256
  char device_name[num_devices][MAX_FILENAME_LEN];
  for (dev = 0;dev < num_devices; dev++) {
    CHECK_ERROR(clGetDeviceInfo(device_ids[dev], CL_DEVICE_NAME, sizeof(device_name[dev]), device_name[dev] ,&size_ret));
  }

  // create a context for the device list
  context = clCreateContext(NULL, num_devices, device_ids, NULL, NULL, &rc);
  CHECK_ERROR_VAL("clCreateContext", rc);

  // create the program
  program = clCreateProgramWithSource(context, 1, (const char **)(&source), NULL, &rc);
  CHECK_ERROR_VAL("clCreateProgramWithSource", rc);

  // done with the source file
  if (source_filename == NULL) {
    // free the memory we allocated
    free(source);
  } else {
    // unmap and close the file
    rc = munmap((void *)source, s_size);
    if (rc != 0) {
      PRINT(stderr, "ERROR: munmap of source failed; returned %d errno=%d\n", rc, errno);
      exit(EXIT_FAILURE);
    }

    rc = close(fin);
    if (rc != 0) {
      PRINT(stderr, "ERROR: close of source failed; returned %d errno=%d\n", rc, errno);
      exit(EXIT_FAILURE);
    }
  }

  // do the build
  rc = clBuildProgram(program, num_devices, device_ids, options, NULL, NULL);

  // retrieve the build status for each device
  cl_build_status build_status;
  int found_an_error = 0;
  for (dev = 0;dev < num_devices; dev++) {
    clGetProgramBuildInfo(program, device_ids[dev], CL_PROGRAM_BUILD_STATUS, sizeof(build_status), &build_status, &size_ret);
    if (build_status == CL_BUILD_ERROR) {
      PRINT(stderr, "clBuildProgram reported CL_BUILD_ERROR for device \"%s\"\n", device_name[dev]);
      found_an_error = 1; // so we exit when we're done showing logs.
    }

    // get the size of the log.
    size_t size_of_log;
    clGetProgramBuildInfo(program, device_ids[dev], CL_PROGRAM_BUILD_LOG, 0, NULL, &size_of_log);

    // if there is a log (more than just the null terminator), get it and dump it
    if (size_of_log > 1) { 
      char *build_log = malloc(size_of_log);
      if (build_log == NULL) {
        PRINT(stderr, "ERROR: malloc for build_log failed\n");
        exit(EXIT_FAILURE);
      }
      clGetProgramBuildInfo(program, device_ids[dev], CL_PROGRAM_BUILD_LOG, size_of_log, build_log, &size_ret);
      PRINT(stderr, "CL_PROGRAM_BUILD_LOG for device \"%s\": \n%s\n", device_name[dev], build_log);
      free(build_log);
    } // there is a log
  } // for num_devices
  CHECK_ERROR_VAL("clBuildProgram", rc);
  if (found_an_error) {
    PRINT(stderr, "CL_BUILD_ERROR. exiting\n");
    exit(EXIT_FAILURE);
  }

  // get and save the binary(s)
  size_t binary_size[num_devices];
  uint total_binary_size = 0;
  unsigned char *binary[num_devices];

  // find out how big each of the binaries are and then get them
  CHECK_ERROR(clGetProgramInfo(program, CL_PROGRAM_BINARY_SIZES, sizeof(binary_size), binary_size, &size_ret));
  for (dev = 0;dev < num_devices; dev++) {
    binary[dev] = malloc(binary_size[dev]);
    if (binary[dev] == NULL) {
      PRINT(stderr, "ERROR: malloc for binary failed\n");
      exit(EXIT_FAILURE);
    }
    total_binary_size += binary_size[dev];
  }
  CHECK_ERROR(clGetProgramInfo(program, CL_PROGRAM_BINARIES, total_binary_size, binary, &size_ret));

  // release the opencl objects we created
  clReleaseProgram(program);
  clReleaseContext(context);

  for (dev = 0;dev < num_devices; dev++) {
    // build the binary filename
    char filename[MAX_FILENAME_LEN] = "";
    char safe_dev_name[MAX_FILENAME_LEN] = "";

    // Convert spaces in device name to underscores
    int i;
    for (i = 0; i < strlen(device_name[dev]); i++) {
      if (device_name[dev][i] == ' ') {
        safe_dev_name[i] = '_';
      } else {
        safe_dev_name[i] = device_name[dev][i];
      }
    }
    if (binary_filename == NULL) {
      // chop off any 'suffix' if there is one
      if (source_filename == NULL) {
        snprintf(filename, MAX_FILENAME_LEN, "%s_%s.ocl", "stdin", safe_dev_name);
      } else {
        char *dot = strrchr(source_filename, '.');
        if (dot != NULL) {
          // null terminate the source string at the point
          *dot = '\0';
          snprintf(filename, MAX_FILENAME_LEN, "%s_%s.ocl", source_filename, safe_dev_name);
          // and restore it so that we can print the name out at the end
          *dot = '.';
        } else {
          snprintf(filename, MAX_FILENAME_LEN, "%s_%s.ocl", source_filename, safe_dev_name);
        }
      }
    } else {
      // append the device name to what they gave us
      snprintf(filename, MAX_FILENAME_LEN, "%s_%s.ocl", binary_filename, safe_dev_name);
    }

    // open and write the binary file out
    FILE *fout = fopen(filename, "w");
    if (fout == NULL) {
      PRINT(stderr, "ERROR: fopen(%s,'w') failed errno=%d\n", filename, errno);
      exit(EXIT_FAILURE);
    }

    rc = fwrite(binary[dev], 1, binary_size[dev], fout);
    if (rc != (int)binary_size[dev]) {
      PRINT(stderr, "ERROR: write(%p, %p, %ld) failed rc=%d errno=%d\n",
                      fout, binary[dev], binary_size[dev], rc, errno);
      exit(EXIT_FAILURE);
    }

    // close the binary file
    rc = fclose(fout);
    if (rc != 0) {
      PRINT(stderr, "ERROR: close failed rc=%d errno=%d\n", rc, errno);
      exit(EXIT_FAILURE);
    }

    PRINT(stdout, "Binary built from %s%s%s for device \"%s\" %s%s%ssaved as file \"%s\"\n",
          source_filename?"source file \"":"stdin", source_filename?source_filename:"", source_filename?"\"":"",
          device_name[dev],
          options?"with options \"":"", options?options:"", options?"\" ":"",
          filename);
  } // for num_devices

  exit(EXIT_SUCCESS);
}
