/**********************************************************************/
/*                                                                    */
/*   Licensed Materials - Property of IBM.                            */
/*   IBM XL C for OpenCL, V0.3 (technology preview)                   */
/*   Copyright IBM Corp. 2007.                                        */
/*   All Rights Reserved.                                             */
/*   US Government Users Restricted Rights -                          */
/*   Use, duplication or disclosure restricted by                     */
/*   GSA ADP Schedule Contract with IBM Corp.                         */
/*                                                                    */
/**********************************************************************/

#ifndef __IBM_VACPP_spu_mcfio_h
#define __IBM_VACPP_spu_mcfio_h 1

#include <spu_intrinsics.h>
#include <stdint.h>

#ifdef __cplusplus
extern "C" {
#endif

/* DMA List Element for MFC List DMA */

typedef struct mfc_list_element {
        uint64_t notify : 1;
        uint64_t reserved : 16;
        uint64_t size : 15;
        uint64_t eal : 32;
} mfc_list_element_t;

#define MFC_MIN_DMA_SIZE_SHIFT  4      /* 16 bytes */
#define MFC_MAX_DMA_SIZE_SHIFT 14      /* 16384 bytes */

#define MFC_MIN_DMA_SIZE (1 << MFC_MIN_DMA_SIZE_SHIFT)
#define MFC_MAX_DMA_SIZE (1 << MFC_MAX_DMA_SIZE_SHIFT)

#define MFC_MIN_DMA_SIZE_MASK (MFC_MIN_DMA_SIZE - 1)
#define MFC_MAX_DMA_SIZE_MASK (MFC_MAX_DMA_SIZE - 1)

#define MFC_MIN_DMA_LIST_SIZE 0x0008   /*   8 bytes */
#define MFC_MAX_DMA_LIST_SIZE 0x4000   /* 16K bytes */

#define MFC_MIN_DMA_LIST_ELEMENTS 0x0001
#define MFC_MAX_DMA_LIST_ELEMENTS 0x0800

#define MFC_BARRIER_ENABLE    0x0001
#define MFC_FENCE_ENABLE      0x0002
#define MFC_LIST_ENABLE       0x0004
#define MFC_RESULT_ENABLE     0x0010

/* Effective Address Utilities */

#define mfc_ea2h(ea)         (uint32_t)((uint64_t)(ea)>>32)
#define mfc_ea2l(ea)         (uint32_t)(ea)
#define mfc_hl2ea(high, low) si_to_ullong(si_selb(si_from_uint(high), si_rotqbyi(si_from_uint(low), -4), si_fsmbi(0x0f0f)))
#define mfc_ceil128(value)   ((value + 127) & ~127)

/* MFC Tag Manager */

/* MFC Tag Manager Mnemonics */
#define MFC_TAG_VALID         0x00000000
#define MFC_TAG_INVALID       0xFFFFFFFF

#define mfc_tag_reserve()                     __mfc_tag_reserve()
#define mfc_tag_release(tag)                  __mfc_tag_release((tag))
#define mfc_multi_tag_reserve(nr_tags)        __mfc_multi_tag_reserve((nr_tags))
#define mfc_multi_tag_release(tag, nr_tags)   __mfc_multi_tag_release((tag),(nr_tags))

extern uint32_t __mfc_tag_reserve(void);
extern uint32_t __mfc_tag_release(uint32_t);
extern uint32_t __mfc_multi_tag_reserve(uint32_t);
extern uint32_t __mfc_multi_tag_release(uint32_t, uint32_t);

/* MFC DMA Commands */

/* MFC DMA Command Mnemonics */
#define MFC_PUT_CMD        0x0020
#define MFC_PUTB_CMD       (MFC_PUT_CMD | MFC_BARRIER_ENABLE)
#define MFC_PUTF_CMD       (MFC_PUT_CMD | MFC_FENCE_ENABLE)
#define MFC_GET_CMD        0x0040
#define MFC_GETB_CMD       (MFC_GET_CMD | MFC_BARRIER_ENABLE)
#define MFC_GETF_CMD       (MFC_GET_CMD | MFC_FENCE_ENABLE)

#define MFC_CMD_WORD(_tid, _rid, _cmd) (((_tid)<<24)|((_rid)<<16)|(_cmd))

#define mfc_put(ls, ea, size, tag, tid, rid)    spu_mfcdma64(ls, mfc_ea2h(ea), mfc_ea2l(ea), size, tag,((tid<<24)|(rid<<16)|MFC_PUT_CMD))
#define mfc_putb(ls, ea, size, tag, tid, rid)   spu_mfcdma64(ls, mfc_ea2h(ea), mfc_ea2l(ea), size, tag,((tid<<24)|(rid<<16)|MFC_PUTB_CMD))
#define mfc_putf(ls, ea, size, tag, tid, rid)   spu_mfcdma64(ls, mfc_ea2h(ea), mfc_ea2l(ea), size, tag,((tid<<24)|(rid<<16)|MFC_PUTF_CMD))
#define mfc_get(ls, ea, size, tag, tid, rid)    spu_mfcdma64(ls, mfc_ea2h(ea), mfc_ea2l(ea), size, tag,((tid<<24)|(rid<<16)|MFC_GET_CMD))
#define mfc_getf(ls, ea, size, tag, tid, rid)   spu_mfcdma64(ls, mfc_ea2h(ea), mfc_ea2l(ea), size,tag,((tid<<24)|(rid<<16)|MFC_GETF_CMD))
#define mfc_getb(ls, ea, size, tag, tid, rid)   spu_mfcdma64(ls, mfc_ea2h(ea), mfc_ea2l(ea), size,tag,((tid<<24)|(rid<<16)|MFC_GETB_CMD))

/* MFC List DMA Commands */

/* MFC List DMA Command Mnemonics */
#define MFC_PUTL_CMD               0x0024
#define MFC_PUTLB_CMD              (MFC_PUTL_CMD | MFC_BARRIER_ENABLE)
#define MFC_PUTLF_CMD              (MFC_PUTL_CMD | MFC_FENCE_ENABLE)
#define MFC_GETL_CMD               0x0044
#define MFC_GETLB_CMD              (MFC_GETL_CMD | MFC_BARRIER_ENABLE)
#define MFC_GETLF_CMD              (MFC_GETL_CMD | MFC_FENCE_ENABLE)

#define mfc_putl(ls, ea, list, list_size, tag, tid, rid)   spu_mfcdma64(ls, mfc_ea2h(ea), (unsigned int)(list), list_size, tag,((tid<<24)|(rid<<16)|MFC_PUTL_CMD))
#define mfc_putlb(ls, ea, list, list_size, tag, tid, rid)  spu_mfcdma64(ls,mfc_ea2h(ea),(unsigned int)(list), list_size, tag,((tid<<24)|(rid<<16)|MFC_PUTLB_CMD))
#define mfc_putlf(ls, ea, list, list_size, tag, tid, rid)  spu_mfcdma64(ls, mfc_ea2h(ea),(unsigned int)(list), list_size, tag,((tid<<24)|(rid<<16)|MFC_PUTLF_CMD))
#define mfc_getl(ls, ea, list, list_size, tag, tid, rid)   spu_mfcdma64(ls,mfc_ea2h(ea),(unsigned int)(list), list_size, tag,((tid<<24)|(rid<<16)|MFC_GETL_CMD))
#define mfc_getlb(ls, ea, list, list_size, tag, tid, rid)  spu_mfcdma64(ls,mfc_ea2h(ea),(unsigned int)(list), list_size, tag,((tid<<24)|(rid<<16)|MFC_GETLB_CMD))
#define mfc_getlf(ls, ea, list, list_size, tag, tid, rid)  spu_mfcdma64(ls,mfc_ea2h(ea),(unsigned int)(list), list_size, tag,((tid<<24)|(rid<<16)|MFC_GETLF_CMD))

/* MFC Atomic Update Commands */

/* MFC Atomic Update Command Mnemonics */
#define MFC_GETLLAR_CMD            0x00D0
#define MFC_PUTLLC_CMD             0x00B4
#define MFC_PUTLLUC_CMD            0x00B0
#define MFC_PUTQLLUC_CMD           0x00B8

#define mfc_getllar(ls, ea, tid, rid)   spu_mfcdma64(ls, mfc_ea2h(ea), mfc_ea2l(ea), 128,   0,((tid<<24)|(rid<<16)|MFC_GETLLAR_CMD))
#define mfc_putllc(ls, ea, tid, rid)        spu_mfcdma64(ls, mfc_ea2h(ea), mfc_ea2l(ea), 128,   0,((tid<<24)|(rid<<16)|MFC_PUTLLC_CMD))
#define mfc_putlluc(ls, ea, tid, rid)       spu_mfcdma64(ls, mfc_ea2h(ea), mfc_ea2l(ea), 128,   0,((tid<<24)|(rid<<16)|MFC_PUTLLUC_CMD))
#define mfc_putqlluc(ls, ea, tag, tid, rid) spu_mfcdma64(ls, mfc_ea2h(ea), mfc_ea2l(ea), 128, tag,((tid<<24)|(rid<<16)|MFC_PUTQLLUC_CMD))

/* MFC SL1 Storage Control Commands */

/* MFC SL1 Storage Control Command Mnemonics */
#define MFC_SDCRT_CMD              0x0080
#define MFC_SDCRTST_CMD            0x0081
#define MFC_SDCRZ_CMD              0x0089
#define MFC_SDCRST_CMD             0x008D
#define MFC_SDCRF_CMD              0x008F

#define mfc_sdcrt(ea, size, tag, tid, rid) spu_mfcdma64(0, mfc_ea2h(ea), mfc_ea2l(ea), size, tag, ((tid<<24)|(rid<<16)|MFC_SDCRT_CMD))
#define mfc_sdcrtst(ea, size, tag, tid, rid) spu_mfcdma64(0, mfc_ea2h(ea), mfc_ea2l(ea), size, tag, ((tid<<24)|(rid<<16)|MFC_SDCRTST_CMD))
#define mfc_sdcrz(ea, size, tag, tid, rid) spu_mfcdma64(0, mfc_ea2h(ea), mfc_ea2l(ea), size, tag, ((tid<<24)|(rid<<16)|MFC_SDCRZ_CMD))
#define mfc_sdcrst(ea, size, tag, tid, rid) spu_mfcdma64(0, mfc_ea2h(ea), mfc_ea2l(ea), size, tag, ((tid<<24)|(rid<<16)|MFC_SDCRST_CMD))
#define mfc_sdcrf(ea, size, tag, tid, rid) spu_mfcdma64(0, mfc_ea2h(ea), mfc_ea2l(ea), size, tag, ((tid<<24)|(rid<<16)|MFC_SDCRF_CMD))

/* MFC Synchronization Commands */

/* MFC Synchronization Command Mnemonics */
#define MFC_SNDSIG_CMD             0x00A0
#define MFC_SNDSIGB_CMD            (MFC_SNDSIGF_CMD | MFC_BARRIER_ENABLE)
#define MFC_SNDSIGF_CMD            (MFC_SNDSIGF_CMD | MFC_FENCE_ENABLE)
#define MFC_BARRIER_CMD            0x00C0
#define MFC_EIEIO_CMD              0x00C8
#define MFC_SYNC_CMD               0x00CC

#define mfc_sndsig(ls, ea, tag, tid, rid)  spu_mfcdma64(ls,mfc_ea2h(ea),mfc_ea2l(ea),4,tag,((tid<<24)|(rid<<16)|MFC_SNDSIG_CMD))
#define mfc_sndsigb(ls, ea, tag, tid, rid) spu_mfcdma64(ls, mfc_ea2h(ea), mfc_ea2l(ea), 4, tag,((tid<<24)|(rid<<16)|MFC_SNDSIGB_CMD))
#define mfc_sndsigf(ls, ea, tag, tid, rid) spu_mfcdma64(ls, mfc_ea2h(ea), mfc_ea2l(ea), 4, tag,((tid<<24)|(rid<<16)|MFC_SNDSIGF_CMD))
#define mfc_barrier(tag)                   spu_mfcdma32(0, 0, 0, tag, MFC_BARRIER_CMD)
#define mfc_eieio(tag, tid, rid)           spu_mfcdma32(0, 0, 0, tag, ((tid<<24)|(rid<<16)|MFC_EIEIO_CMD))
#define mfc_sync(tag)                      spu_mfcdma32(0, 0, 0, tag, MFC_SYNC_CMD)

/* MFC DMA Status */

#define mfc_stat_cmd_queue() spu_readchcnt(MFC_Cmd)
#define mfc_write_tag_mask(mask) spu_writech(MFC_WrTagMask, mask)
#define mfc_read_tag_mask() spu_readch(MFC_RdTagMask)

/* MFC Write Tag Update Conditions */
#define MFC_TAG_UPDATE_IMMEDIATE   0x0
#define MFC_TAG_UPDATE_ANY         0x1
#define MFC_TAG_UPDATE_ALL         0x2

#define mfc_write_tag_update(ts)         spu_writech(MFC_WrTagUpdate, ts)
#define mfc_write_tag_update_immediate() spu_writech(MFC_WrTagUpdate, MFC_TAG_UPDATE_IMMEDIATE)
#define mfc_write_tag_update_any()       spu_writech(MFC_WrTagUpdate, MFC_TAG_UPDATE_ANY)
#define mfc_write_tag_update_all()       spu_writech(MFC_WrTagUpdate, MFC_TAG_UPDATE_ALL)
#define mfc_stat_tag_update()            spu_readchcnt(MFC_WrTagUpdate)
#define mfc_read_tag_status()            spu_readch(MFC_RdTagStat)
#define mfc_read_tag_status_immediate()  spu_mfcstat(MFC_TAG_UPDATE_IMMEDIATE)
#define mfc_read_tag_status_any()        spu_mfcstat(MFC_TAG_UPDATE_ANY)
#define mfc_read_tag_status_all()        spu_mfcstat(MFC_TAG_UPDATE_ALL)
#define mfc_stat_tag_status()            spu_readchcnt(MFC_RdTagStat)
#define mfc_read_list_stall_status()     spu_readch(MFC_RdListStallStat)
#define mfc_stat_list_stall_status()     spu_readchcnt(MFC_RdListStallStat)
#define mfc_write_list_stall_ack(tag)    spu_writech(MFC_WrListStallAck, tag)


/* Read Atomic Command Status Result */
#define MFC_PUTLLC_STATUS    0x00000001
#define MFC_PUTLLUC_STATUS   0x00000002
#define MFC_GETLLAR_STATUS   0x00000004

#define mfc_read_atomic_status()         spu_readch(MFC_RdAtomicStat)
#define mfc_stat_atomic_status()         spu_readchcnt(MFC_RdAtomicStat)

/* MFC Multisource Synchronization Request */

#define mfc_write_multi_src_sync_request() spu_writech(MFC_WrMSSyncReq,0)
#define mfc_stat_multi_src_sync_request()  spu_readchcnt(MFC_WrMSSyncReq)

/* SPU Signal Notification */

#define spu_read_signal1()               spu_readch(SPU_RdSigNotify1)
#define spu_stat_signal1()               spu_readchcnt(SPU_RdSigNotify1)
#define spu_read_signal2()               spu_readch(SPU_RdSigNotify2)
#define spu_stat_signal2()               spu_readchcnt(SPU_RdSigNotify2)

/* SPU Mailboxes */

#define spu_read_in_mbox()               spu_readch(SPU_RdInMbox)
#define spu_stat_in_mbox()               spu_readchcnt(SPU_RdInMbox)
#define spu_write_out_mbox(data)         spu_writech(SPU_WrOutMbox, data)
#define spu_stat_out_mbox()              spu_readchcnt(SPU_WrOutMbox)
#define spu_write_out_intr_mbox(data)    spu_writech(SPU_WrOutIntrMbox, data)
#define spu_stat_out_intr_mbox()         spu_readchcnt(SPU_WrOutIntrMbox)

/* SPU Decrementer */

#define spu_read_decrementer()           spu_readch(SPU_RdDec)
#define spu_write_decrementer(count)     spu_writech(SPU_WrDec, count)

/* SPU Event */

/* MFC Event Bit-Fields */
#define MFC_MULTI_SRC_SYNC_EVENT           0x1000
#define MFC_PRIV_ATTN_EVENT                0x0800
#define MFC_LLR_LOST_EVENT                 0x0400
#define MFC_SIGNAL_NOTIFY_1_EVENT          0x0200
#define MFC_SIGNAL_NOTIFY_2_EVENT          0x0100
#define MFC_OUT_MBOX_AVAILABLE_EVENT       0x0080
#define MFC_OUT_INTR_MBOX_AVAILABLE_EVENT  0x0040
#define MFC_DECREMENTER_EVENT              0x0020
#define MFC_IN_MBOX_AVAILABLE_EVENT        0x0010
#define MFC_COMMAND_QUEUE_AVAILABLE_EVENT  0x0008
#define MFC_LIST_STALL_NOTIFY_EVENT        0x0002
#define MFC_TAG_STATUS_UPDATE_EVENT        0x0001

#define spu_read_event_status()          spu_readch(SPU_RdEventStat)
#define spu_stat_event_status()          spu_readchcnt(SPU_RdEventStat)
#define spu_write_event_mask(mask)       spu_writech(SPU_WrEventMask, mask)
#define spu_write_event_ack(ack)         spu_writech(SPU_WrEventAck, ack)
#define spu_read_event_mask()            spu_readch(SPU_RdEventMask)

/* SPU State Management */

#define spu_read_machine_status()        spu_readch(SPU_RdMachStat)
#define spu_write_srr0(srr0)             spu_writech(SPU_WrSRR0,srr0)
#define spu_read_srr0()                  spu_readch(SPU_RdSRR0)

/* Interrupt Safe Critical Sections */

#ifndef __cplusplus
static __inline__ uint32_t mfc_begin_critical_section() __attribute__((always_inline));
#endif

static __inline__ uint32_t mfc_begin_critical_section() {
 #ifdef SPU_MFCIO_INTERRUPT_SAFE
  uint32_t __mach_stat;
  __mach_stat = spu_readch(SPU_RdMachStat); 
  spu_idisable();
  return __mach_stat;
 #else
  return 0;
 #endif
}

#ifndef __cplusplus
static __inline__ void mfc_end_critical_section(uint32_t __mach_stat) __attribute__((always_inline));
#endif

static __inline__ void mfc_end_critical_section(uint32_t __mach_stat __attribute__((__unused__))) {
 #ifdef SPU_MFCIO_INTERRUPT_SAFE
  if ((__mach_stat) & 1) spu_ienable();
 #endif
}

#ifdef __cplusplus
}
#endif

#endif /* __IBM_VACPP_spu_mcfio_h */
